% 2D Time Reversal Reconstruction For A Rectangular Detector with 
% Directional Elements
%
% This example demonstrates the effects on time reversal imaging of using
% directional sensor elements. 
%
% author: Bradley Treeby & Ben Cox
% date: 18th January 2010
% last update: 20th January 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
PML_size = 20;          % size of the PML in pixels
Nx = 256 - 2*PML_size;  % number of pixels in the x (column) direction
Nz = 128 - 2*PML_size;  % number of pixels in the z (row) direction
dx = 0.1e-3;            % pixel width [m]
dz = 0.1e-3;            % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;	% [m/s]
medium.density = 1000;      % [kg/m^3]

% create initial pressure distribution using makeDisc
disc_magnitude = 5;
disc_x_pos = 140;   % pixels
disc_z_pos = 60;    % pixels
disc_radius = 5;    % pixels
disc_2 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

disc_x_pos = 110;   % pixels
disc_z_pos = 30;    % pixels
disc_radius = 8;    % pixels
disc_1 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

% smooth the initial pressure distribution and restore the magnitude
p0 = smooth(kgrid,disc_1 + disc_2, true);

% assign to the source structure
source.p0 = p0;

% define a four-sided, square sensor
sensor.mask = zeros(kgrid.Nz, kgrid.Nx);
sensor.mask(1, :) = 1;
sensor.mask(Nz, :) = 1;
sensor.mask(:, 1) = 1;
sensor.mask(:, Nx) = 1;

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed);

% set the input arguements: force the PML to be outside the computational
% grid; switch off p0 smoothing within kspaceFirstOrder2D (as the grids for
% c and  rho in this example are homogeneous, the remaining smoothing
% parameters are not used so they can all be turned off)
input_args = {'PMLInside', false, 'PMLSize', PML_size, 'Smooth', false};

% run the simulation for omnidirectional detector elements
sensor_data = kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});

% define the directionality of the sensor elements
sensor.directivity_angle = zeros(kgrid.Nz,kgrid.Nx);
sensor.directivity_angle(1, :) = 0;     % max sensitivity in z direction
sensor.directivity_angle(Nz, :) = 0;    % max sensitivity in z direction
sensor.directivity_angle(:, 1) = pi/2;  % max sensitivity in x direction
sensor.directivity_angle(:, Nx) = pi/2; % max sensitivity in x direction

% define the directivity size
sensor.directivity_size = 20*kgrid.dx;

% run the simulation with directional elements
sensor_data_directional = kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});

% reset the initial pressure
source.p0 = 0;

% assign the time reversal data for the omnidirectional case
sensor.time_reversal_boundary_data = sensor_data;

% run the time reversal reconstruction
p0_recon = kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});

% assign the time reversal data for the directional case
sensor.time_reversal_boundary_data = sensor_data_directional;

% run the time reversal reconstruction with directional elements
p0_recon_directional = kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the initial pressure and sensor distribution
figure;
imagesc(kgrid.x(1,:)*1e3, kgrid.z(:,1)*1e3, p0 + sensor.mask*disc_magnitude, [-disc_magnitude disc_magnitude]);
colormap(getColorMap);
ylabel('z-position [mm]');
xlabel('x-position [mm]');
axis image;
colorbar;

% plot the reconstructed initial pressure 
figure;
imagesc(kgrid.x(1,:)*1e3, kgrid.z(:,1)*1e3, p0_recon, [-disc_magnitude disc_magnitude]);
colormap(getColorMap);
ylabel('z-position [mm]');
xlabel('x-position [mm]');
axis image;
colorbar;

% plot the reconstructed initial pressure with directivity
figure;
imagesc(kgrid.x(1,:)*1e3, kgrid.z(:,1)*1e3, p0_recon_directional, [-disc_magnitude disc_magnitude]);
colormap(getColorMap);
ylabel('z-position [mm]');
xlabel('x-position [mm]');
axis image;
colorbar;

% plot a profile for comparison
figure;
plot(kgrid.x(1,:)*1e3, p0(disc_z_pos, :), 'k-', kgrid.x(1,:)*1e3, p0_recon(disc_z_pos, :), 'r--',  kgrid.x(1,:)*1e3, p0_recon_directional(disc_z_pos, :), 'b:');
xlabel('x-position [mm]');
ylabel('Pressure');
legend('True', 'Omnidirectional','Directional');
axis tight;
set(gca, 'YLim', [0 5.1]);
